import pygame
import sys
import math
import random

from player import Player
from world import WorldMap
from raycaster import Raycaster
from enemy import Enemy
from projectile import Projectile
from hud import HUD
from deathscreen import DeathScreen

WIDTH, HEIGHT = 800, 600
HALF_HEIGHT = HEIGHT // 2


class Game:
    def __init__(self):
        pygame.init()
        self.screen = pygame.display.set_mode((WIDTH, HEIGHT))
        pygame.display.set_caption("DUDE")

        self.clock = pygame.time.Clock()

        self.world = WorldMap("map_layout.txt")
        self.player = Player(2.5, 2.5)
        self.raycaster = Raycaster(self.world)

        self.enemies = [
            Enemy(5.5, 5.5),
            Enemy(8.0, 3.0),
        ]

        self.projectiles = []

        # HUD / UI
        self.hud = HUD(self.screen)
        self.deathscreen = DeathScreen(self.screen)

        # Score system
        self.score = 0
        self.kills = 0

        # Respawn system
        self.respawn_delay = 3.0
        self.dead_enemies = {}  # enemy -> death_time

    # --------------------
    # Respawn helper
    # --------------------
    def get_random_spawn_position(self):
        while True:
            x = random.randint(1, WIDTH - 2) + 0.5
            y = random.randint(1, HEIGHT - 2) + 0.5

            if not self.world.check_collision(x, y):
                return x, y

    # --------------------
    # Update
    # --------------------
    def update(self, dt):
        keys = pygame.key.get_pressed()

        if keys[pygame.K_ESCAPE]:
            pygame.quit()
            sys.exit()

        # --------------------
        # Death state
        # --------------------
        if not self.player.alive:
            if keys[pygame.K_r]:
                self.__init__()  # kompletter Reset
            return

        # --------------------
        # Player
        # --------------------
        self.player.update(keys, self.world)

        if self.player.weapon.fired_this_frame:
            self.spawn_projectile()

        # --------------------
        # Projectiles
        # --------------------
        for p in self.projectiles:
            p.update(self.world, dt)

            for e in self.enemies:
                if not e.alive or not p.alive:
                    continue

                dist = math.hypot(e.x - p.x, e.y - p.y)
                if dist < e.radius + p.radius:
                    if self.raycaster.is_visible(self.player, e.x, e.y):
                        killed = e.take_damage(self.player.weapon.damage)
                        p.alive = False

                        if killed and e not in self.dead_enemies:
                            self.kills += 1
                            self.score += 100
                            self.dead_enemies[e] = (
                                pygame.time.get_ticks() / 1000.0
                            )

        self.projectiles = [p for p in self.projectiles if p.alive]

        # --------------------
        # Enemies
        # --------------------
        for e in self.enemies:
            if e.alive:
                e.update(self.player, self.world, dt)

        # --------------------
        # Respawning
        # --------------------
        current_time = pygame.time.get_ticks() / 1000.0
        for enemy, death_time in list(self.dead_enemies.items()):
            if current_time - death_time >= self.respawn_delay:
                x, y = self.get_random_spawn_position()
                enemy.respawn(x, y)
                del self.dead_enemies[enemy]

    # --------------------
    # Helpers
    # --------------------
    def spawn_projectile(self):
        self.projectiles.append(
            Projectile(self.player.x, self.player.y, self.player.angle)
        )

    # --------------------
    # Draw
    # --------------------
    def draw(self):
        self.screen.fill((0, 0, 0))
        pygame.draw.rect(self.screen, (100, 100, 100), (0, 0, WIDTH, HALF_HEIGHT))
        pygame.draw.rect(self.screen, (40, 40, 40), (0, HALF_HEIGHT, WIDTH, HALF_HEIGHT))

        self.raycaster.cast_rays(self.screen, self.player)

        self.raycaster.render_sprites(
            self.screen,
            self.player,
            self.enemies,
            self.projectiles
        )

        self.player.weapon.draw(self.screen)

        # HUD
        self.hud.draw_healthbar(self.player)
        self.hud.draw_score(self.score, self.kills)

        # Death Screen Overlay
        if not self.player.alive:
            self.deathscreen.draw(self.score, self.kills)

    # --------------------
    # Main Loop
    # --------------------
    def run(self):
        while True:
            dt = self.clock.tick(60) / 1000.0

            for event in pygame.event.get():
                if event.type == pygame.QUIT:
                    pygame.quit()
                    sys.exit()

            self.update(dt)
            self.draw()
            pygame.display.flip()
